<?php
require_once __DIR__ . '/db.php';

function wallet_get_balance(int $userId) : float {
  $pdo = db();
  $stmt = $pdo->prepare("SELECT balance FROM wallets WHERE user_id=?");
  $stmt->execute([$userId]);
  $row = $stmt->fetch();
  return $row ? (float)$row['balance'] : 0.0;
}

function wallet_credit(int $userId, float $amount, string $reason='TOPUP', string $refType=null, int $refId=null) : void {
  if ($amount <= 0) throw new Exception("Invalid amount");
  $pdo = db();
  $pdo->beginTransaction();
  try {
    // Lock wallet row
    $stmt = $pdo->prepare("SELECT id, balance FROM wallets WHERE user_id=? FOR UPDATE");
    $stmt->execute([$userId]);
    $w = $stmt->fetch();
    if (!$w) {
      $pdo->prepare("INSERT INTO wallets(user_id,balance,created_at,updated_at) VALUES(?,?,NOW(),NOW())")
          ->execute([$userId, 0]);
      $stmt->execute([$userId]);
      $w = $stmt->fetch();
    }
    $opening = (float)$w['balance'];
    $closing = $opening + $amount;

    $pdo->prepare("UPDATE wallets SET balance=?, updated_at=NOW() WHERE user_id=?")->execute([$closing, $userId]);
    $pdo->prepare("INSERT INTO wallet_ledger(user_id,type,amount,reason,reference_type,reference_id,opening_balance,closing_balance,created_at) 
                   VALUES(?,?,?,?,?,?,?, ?, NOW())")
        ->execute([$userId,'CREDIT',$amount,$reason,$refType,$refId,$opening,$closing]);
    $pdo->commit();
  } catch(Exception $e){
    $pdo->rollBack();
    throw $e;
  }
}

function wallet_debit(int $userId, float $amount, string $reason='SERVICE_CHARGE', string $refType=null, int $refId=null) : void {
  if ($amount <= 0) throw new Exception("Invalid amount");
  $pdo = db();
  $pdo->beginTransaction();
  try {
    $stmt = $pdo->prepare("SELECT id, balance FROM wallets WHERE user_id=? FOR UPDATE");
    $stmt->execute([$userId]);
    $w = $stmt->fetch();
    if (!$w) throw new Exception("Wallet not found");
    $opening = (float)$w['balance'];
    if ($opening < $amount) throw new Exception("Insufficient wallet balance");
    $closing = $opening - $amount;

    $pdo->prepare("UPDATE wallets SET balance=?, updated_at=NOW() WHERE user_id=?")->execute([$closing, $userId]);
    $pdo->prepare("INSERT INTO wallet_ledger(user_id,type,amount,reason,reference_type,reference_id,opening_balance,closing_balance,created_at) 
                   VALUES(?,?,?,?,?,?,?, ?, NOW())")
        ->execute([$userId,'DEBIT',$amount,$reason,$refType,$refId,$opening,$closing]);
    $pdo->commit();
  } catch(Exception $e){
    $pdo->rollBack();
    throw $e;
  }
}
